<?php

namespace App\Http\Controllers;

use App\Models\ClassModel;
use App\Models\Subject;
use App\Models\Student;
use App\Models\TeachingJurnal;
use Illuminate\Http\Request;
use Carbon\Carbon;

class TeacherController extends Controller
{
    public function myClasses()
    {
        $npsn = auth()->user()->npsn;
        $classes = ClassModel::where('npsn', $npsn)->get(); 
        if ($classes->isEmpty()) {
            // Fallback if no npsn assigned yet or just return all for now if school system is loose
            $classes = ClassModel::all();
        }
        return response()->json(['data' => $classes]);
    }

    public function mySubjects()
    {
        $npsn = auth()->user()->npsn;
        $subjects = Subject::with('class')
            ->where(function($q) use ($npsn) {
                $q->where('npsn', $npsn)
                  ->orWhereHas('class', function($q2) use ($npsn) {
                      $q2->where('npsn', $npsn);
                  });
            })->get();

        if ($subjects->isEmpty()) {
            $subjects = Subject::with('class')->get();
        }
        return response()->json(['data' => $subjects]);
    }

    public function storeSubject(Request $request)
    {
        $request->validate([
            'name'     => 'required|string',
            'code'     => 'nullable|string',
            'description' => 'nullable|string',
            'class_id' => 'required|exists:classes,id',
        ]);

        $subject = Subject::create($request->all() + ['npsn' => auth()->user()->npsn]);
        return response()->json(['data' => $subject], 201);
    }

    public function updateSubject(Request $request, Subject $subject)
    {
        $request->validate([
            'name'     => 'required|string',
            'code'     => 'nullable|string',
            'description' => 'nullable|string',
            'class_id' => 'required|exists:classes,id',
        ]);

        $subject->update($request->all());
        return response()->json(['data' => $subject]);
    }

    public function destroySubject(Subject $subject)
    {
        $subject->delete();
        return response()->json(['message' => 'Mata pelajaran berhasil dihapus']);
    }

    public function myStudents(Request $request)
    {
        $classId = $request->query('class_id');
        if (!$classId) {
            return response()->json(['data' => []]);
        }
        $students = Student::where('class_id', $classId)->get();
        return response()->json(['data' => $students]);
    }

    public function dashboardStats()
    {
        $user = auth()->user();
        $npsn = $user->npsn;
        $today = Carbon::today();

        $totalJournals = TeachingJurnal::where('teacher_id', $user->id)->count();
        $todayJournals = TeachingJurnal::where('teacher_id', $user->id)
            ->whereDate('date', $today)
            ->count();
        
        $activeStudents = Student::whereHas('class', function($q) use ($npsn) {
            $q->where('npsn', $npsn);
        })->count();

        if ($activeStudents === 0) $activeStudents = Student::count();

        // Calculate avg attendance from their journals
        $journals = TeachingJurnal::where('teacher_id', $user->id)->get();
        $totalStudentsSeen = 0;
        $totalHadir = 0;

        foreach ($journals as $journal) {
            if ($journal->absent_list) {
                $list = $journal->absent_list;
                $totalStudentsSeen += count($list);
                foreach ($list as $status) {
                    if ($status === 'Hadir') {
                        $totalHadir++;
                    }
                }
            }
        }

        $avgAttendance = $totalStudentsSeen > 0 ? round(($totalHadir / $totalStudentsSeen) * 100) : 0;

        return response()->json(['data' => [
            'totalJournals' => $totalJournals,
            'todayJournals' => $todayJournals,
            'activeStudents' => $activeStudents,
            'avgAttendance' => $avgAttendance,
            'journals' => $journals 
        ]]);
    }
}
