<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\TeachingJurnal;
use App\Models\Student;
use App\Models\ClassModel;
use Illuminate\Support\Facades\DB;

class ReportingController extends Controller
{
    public function attendanceReport(Request $request)
    {
        $npsn = auth()->user()->npsn;
        $dateFrom = $request->startDate ?? now()->startOfMonth();
        $dateTo = $request->endDate ?? now()->endOfMonth();
        $classId = $request->classId;

        // Get journals in range
        $query = TeachingJurnal::where('npsn', $npsn)
            ->whereBetween('date', [$dateFrom, $dateTo]);
        
        if ($classId) {
            $query->where('class_id', $classId);
        }

        $journals = $query->get();

        // Get students in class (or all if no class selected)
        $studentQuery = Student::whereHas('class', function($q) use ($npsn) {
            $q->where('npsn', $npsn);
        });
        if ($classId) {
            $studentQuery->where('class_id', $classId);
        }
        $students = $studentQuery->get();

        $report = $students->map(function ($student) use ($journals) {
            $hadir = 0;
            $sakit = 0;
            $izin = 0;
            $alpa = 0;
            $totalDays = 0;

            foreach ($journals as $journal) {
                if ($journal->absent_list && isset($journal->absent_list[$student->id])) {
                    $status = $journal->absent_list[$student->id];
                    if ($status === 'Hadir') $hadir++;
                    elseif ($status === 'Sakit') $sakit++;
                    elseif ($status === 'Izin') $izin++;
                    elseif ($status === 'Alfa') $alpa++;
                    $totalDays++;
                }
            }

            return [
                'name' => $student->name,
                'hadir' => $hadir,
                'sakit' => $sakit,
                'izin' => $izin,
                'alpa' => $alpa,
                'percentage' => $totalDays > 0 ? round(($hadir / $totalDays) * 100) : 0
            ];
        });

        return response()->json(['data' => $report->values()]);
    }

    public function teachingHoursReport(Request $request)
    {
        $npsn = auth()->user()->npsn;
        $dateFrom = $request->startDate ?? now()->startOfMonth();
        $dateTo = $request->endDate ?? now()->endOfMonth();

        $journals = TeachingJurnal::where('npsn', $npsn)
            ->whereBetween('date', [$dateFrom, $dateTo])
            ->with(['teacher', 'subject'])
            ->get();

        $report = $journals->groupBy('teacher_id')->map(function ($items) {
            $teacher = $items->first()->teacher;
            $subjects = $items->pluck('subject.name')->unique()->values();
            
            return [
                'teacher_name' => $teacher->name ?? 'Unknown',
                'total_journals' => $items->count(),
                'subjects' => $subjects
            ];
        });

        return response()->json(['data' => $report->values()]);
    }

    public function supervisionReport(Request $request)
    {
        $npsn = auth()->user()->npsn;
        $dateFrom = $request->startDate ?? now()->startOfMonth();
        $dateTo = $request->endDate ?? now()->endOfMonth();

        $supervision = TeachingJurnal::where('npsn', $npsn)
            ->whereBetween('date', [$dateFrom, $dateTo])
            ->whereHas('teacher', function($q) {
                $q->where('role', 'guru');
            })
            ->with(['teacher', 'class', 'subject'])
            ->orderBy('date', 'desc')
            ->get();

        return response()->json(['data' => $supervision]);
    }
}