<?php

namespace App\Http\Controllers;

use App\Models\TeachingJurnal;
use Illuminate\Http\Request;
use Barryvdh\DomPDF\Facade\Pdf;

class JurnalController extends Controller
{
    public function index(Request $request)
    {
        $jurnals = TeachingJurnal::with(['teacher', 'class', 'subject'])
            ->when(
                $request->user()->role === 'guru',
                fn($q) => $q->where('teacher_id', $request->user()->id)
            )
            ->where('npsn', $request->user()->npsn)
            ->orderBy('date', 'desc')
            ->paginate(10);

        return response()->json($jurnals);
    }

    public function store(Request $request)
    {
        if ($request->user()->role !== 'guru') {
            return response()->json(['message' => 'Admin tidak diperbolehkan membuat jurnal'], 403);
        }

        $request->validate([
            'class_id' => 'required|exists:classes,id',
            'subject_id' => 'required|exists:subjects,id',
            'date' => 'required|date',
            'hour_start' => 'required',
            'hour_end' => 'required',
            'material' => 'required',
            'absent_list' => 'nullable|array',
            'notes' => 'nullable|string'
        ]);

        $journal = TeachingJurnal::create([
            'teacher_id' => $request->user()->id,
            'class_id' => $request->class_id,
            'subject_id' => $request->subject_id,
            'date' => $request->date,
            'hour_start' => $request->hour_start,
            'hour_end' => $request->hour_end,
            'material' => $request->material,
            'absent_list' => $request->absent_list,
            'notes' => $request->notes,
            'npsn' => $request->user()->npsn,
        ]);

        return response()->json(['data' => $journal], 201);
    }

    public function show(TeachingJurnal $journal)
    {
        $journal->load(['teacher', 'class', 'subject']);
        return response()->json(['data' => $journal]);
    }

    public function update(Request $request, TeachingJurnal $journal)
    {
        if ($request->user()->role === 'guru' && $journal->teacher_id !== $request->user()->id) {
            return response()->json(['message' => 'Tidak boleh mengedit jurnal guru lain'], 403);
        }

        $request->validate([
            'class_id' => 'required|exists:classes,id',
            'subject_id' => 'required|exists:subjects,id',
            'date' => 'required|date',
            'hour_start' => 'required',
            'hour_end' => 'required',
            'material' => 'required',
            'absent_list' => 'nullable|array',
            'notes' => 'nullable|string'
        ]);

        $journal->update($request->all());
        return response()->json(['data' => $journal]);
    }

    public function destroy(Request $request, TeachingJurnal $journal)
    {
        if ($request->user()->role === 'guru' && $journal->teacher_id !== $request->user()->id) {
            return response()->json(['message' => 'Tidak boleh menghapus jurnal guru lain'], 403);
        }

        $journal->delete();
        return response()->json(['message' => 'Jurnal dihapus']);
    }

    public function exportPdf($id)
    {
        $journal = TeachingJurnal::with(['teacher', 'class', 'subject'])->findOrFail($id);
        $pdf = Pdf::loadView('pdf.journal', compact('journal'));
        return $pdf->download("jurnal_{$journal->date}.pdf");
    }
}