<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Imports\StudentsImport;
use App\Models\TeachingJurnal;
use App\Models\Student;
use Maatwebsite\Excel\Facades\Excel;
use Barryvdh\DomPDF\Facade\Pdf;
use App\Exports\JournalsExport;
use App\Exports\AttendanceExport;
use App\Exports\TeachingHoursExport;

class ImportExportController extends Controller
{
    // --- Import Siswa ---
    public function importStudents(Request $request)
    {
        $request->validate([
            'file' => 'required|mimes:xlsx,xls,csv',
        ]);

        try {
            $import = new StudentsImport;
            Excel::import($import, $request->file('file'));

            return response()->json([
                'message' => 'Data siswa berhasil diimpor'
            ]);
        } catch (\Maatwebsite\Excel\Validators\ValidationException $e) {
            $failures = $e->failures();
            $errors = [];
            foreach ($failures as $failure) {
                $errors[] = [
                    'row' => $failure->row(),
                    'attribute' => $failure->attribute(),
                    'errors' => $failure->errors(),
                ];
            }
            return response()->json([
                'message' => 'Terdapat kesalahan validasi',
                'errors' => $errors
            ], 422);
        } catch (\Exception $e) {
            return response()->json([
                'message' => 'Gagal mengimpor data: ' . $e->getMessage()
            ], 500);
        }
    }

    // --- Journal Exports ---
    public function exportJournalsExcel(Request $request)
    {
        $npsn = auth()->user()->npsn;
        $fileName = 'jurnal_mengajar_' . now()->format('YmdHis') . '.xlsx';
        return Excel::download(new JournalsExport($npsn, $request->startDate, $request->endDate), $fileName);
    }

    public function exportJournalsPdf(Request $request)
    {
        $npsn = auth()->user()->npsn;
        $startDate = $request->startDate ?? now()->startOfMonth()->format('Y-m-d');
        $endDate = $request->endDate ?? now()->endOfMonth()->format('Y-m-d');

        $journals = TeachingJurnal::with(['teacher','class','subject'])
            ->where('npsn', $npsn)
            ->whereBetween('date', [$startDate, $endDate])
            ->orderBy('date', 'asc')
            ->get();

        $pdf = Pdf::loadView('pdf.journals', compact('journals', 'startDate', 'endDate'));
        return $pdf->download('jurnal_mengajar_' . $startDate . '_to_' . $endDate . '.pdf');
    }

    // --- Attendance Exports ---
    public function exportAttendanceExcel(Request $request)
    {
        $npsn = auth()->user()->npsn;
        $fileName = 'rekap_absensi_' . now()->format('YmdHis') . '.xlsx';
        return Excel::download(new AttendanceExport($npsn, $request->startDate, $request->endDate, $request->classId), $fileName);
    }

    public function exportAttendancePdf(Request $request)
    {
        $npsn = auth()->user()->npsn;
        $startDate = $request->startDate ?? now()->startOfMonth()->format('Y-m-d');
        $endDate = $request->endDate ?? now()->endOfMonth()->format('Y-m-d');
        $classId = $request->classId;

        $query = TeachingJurnal::where('npsn', $npsn)->whereBetween('date', [$startDate, $endDate]);
        if ($classId) $query->where('class_id', $classId);
        $journals = $query->get();

        $studentQuery = Student::where('npsn', $npsn);
        if ($classId) $studentQuery->where('class_id', $classId);
        $students = $studentQuery->get();

        $report = $students->map(function ($student) use ($journals) {
            $hadir = 0; $sakit = 0; $izin = 0; $alpa = 0; $totalDays = 0;
            foreach ($journals as $journal) {
                if ($journal->absent_list && isset($journal->absent_list[$student->id])) {
                    $status = $journal->absent_list[$student->id];
                    if ($status === 'Hadir') $hadir++;
                    elseif ($status === 'Sakit') $sakit++;
                    elseif ($status === 'Izin') $izin++;
                    elseif ($status === 'Alfa') $alpa++;
                    $totalDays++;
                }
            }
            return [
                'name' => $student->name,
                'nis' => $student->nis,
                'class_name' => $student->class->name ?? 'N/A',
                'hadir' => $hadir,
                'sakit' => $sakit,
                'izin' => $izin,
                'alpa' => $alpa,
                'percentage' => ($totalDays > 0 ? round(($hadir / $totalDays) * 100) : 0) . '%'
            ];
        })->values()->toArray();

        $pdf = Pdf::loadView('pdf.attendance', compact('report', 'startDate', 'endDate'));
        return $pdf->download('rekap_absensi_' . $startDate . '_to_' . $endDate . '.pdf');
    }

    // --- Teaching Hours Exports ---
    public function exportTeachingHoursExcel(Request $request)
    {
        $npsn = auth()->user()->npsn;
        $fileName = 'jam_mengajar_' . now()->format('YmdHis') . '.xlsx';
        return Excel::download(new TeachingHoursExport($npsn, $request->startDate, $request->endDate), $fileName);
    }

    public function exportTeachingHoursPdf(Request $request)
    {
        $npsn = auth()->user()->npsn;
        $startDate = $request->startDate ?? now()->startOfMonth()->format('Y-m-d');
        $endDate = $request->endDate ?? now()->endOfMonth()->format('Y-m-d');

        $journals = TeachingJurnal::where('npsn', $npsn)
            ->whereBetween('date', [$startDate, $endDate])
            ->with(['teacher', 'subject', 'class'])
            ->get();

        $report = $journals->groupBy('teacher_id')->map(function ($items) {
            $teacher = $items->first()->teacher;
            $subjects = $items->pluck('subject.name')->unique()->implode(', ');
            return [
                'teacher_name' => $teacher->name ?? 'Unknown',
                'total_journals' => $items->count(),
                'subjects' => $subjects
            ];
        })->values()->toArray();

        $pdf = Pdf::loadView('pdf.teaching_hours', compact('report', 'startDate', 'endDate'));
        return $pdf->download('jam_mengajar_' . $startDate . '_to_' . $endDate . '.pdf');
    }
}