import { AutomationEvent, Param, ParamOptions } from "../context/Param";
import { Seconds, Ticks, Time, UnitMap, UnitName } from "../type/Units";
import { Timeline } from "../util/Timeline";
declare type TickAutomationEvent = AutomationEvent & {
    ticks: number;
};
interface TickParamOptions<TypeName extends UnitName> extends ParamOptions<TypeName> {
    multiplier: number;
}
/**
 * A Param class just for computing ticks. Similar to the {@link Param} class,
 * but offers conversion to BPM values as well as ability to compute tick
 * duration and elapsed ticks
 */
export declare class TickParam<TypeName extends "hertz" | "bpm"> extends Param<TypeName> {
    readonly name: string;
    /**
     * The timeline which tracks all of the automations.
     */
    protected _events: Timeline<TickAutomationEvent>;
    /**
     * The internal holder for the multiplier value
     */
    private _multiplier;
    /**
     * @param param The AudioParam to wrap
     * @param units The unit name
     * @param convert Whether or not to convert the value to the target units
     */
    /**
     * @param value The initial value of the signal
     */
    constructor(value?: number);
    constructor(options: Partial<TickParamOptions<TypeName>>);
    static getDefaults(): TickParamOptions<any>;
    setTargetAtTime(value: UnitMap[TypeName], time: Time, constant: number): this;
    setValueAtTime(value: UnitMap[TypeName], time: Time): this;
    linearRampToValueAtTime(value: UnitMap[TypeName], time: Time): this;
    exponentialRampToValueAtTime(value: UnitMap[TypeName], time: Time): this;
    /**
     * Returns the tick value at the time. Takes into account
     * any automation curves scheduled on the signal.
     * @param  event The time to get the tick count at
     * @return The number of ticks which have elapsed at the time given any automations.
     */
    private _getTicksUntilEvent;
    /**
     * Returns the tick value at the time. Takes into account
     * any automation curves scheduled on the signal.
     * @param  time The time to get the tick count at
     * @return The number of ticks which have elapsed at the time given any automations.
     */
    getTicksAtTime(time: Time): Ticks;
    /**
     * Return the elapsed time of the number of ticks from the given time
     * @param ticks The number of ticks to calculate
     * @param  time The time to get the next tick from
     * @return The duration of the number of ticks from the given time in seconds
     */
    getDurationOfTicks(ticks: Ticks, time: Time): Seconds;
    /**
     * Given a tick, returns the time that tick occurs at.
     * @return The time that the tick occurs.
     */
    getTimeOfTick(tick: Ticks): Seconds;
    /**
     * Convert some number of ticks their the duration in seconds accounting
     * for any automation curves starting at the given time.
     * @param  ticks The number of ticks to convert to seconds.
     * @param  when  When along the automation timeline to convert the ticks.
     * @return The duration in seconds of the ticks.
     */
    ticksToTime(ticks: Ticks, when: Time): Seconds;
    /**
     * The inverse of {@link ticksToTime}. Convert a duration in
     * seconds to the corresponding number of ticks accounting for any
     * automation curves starting at the given time.
     * @param  duration The time interval to convert to ticks.
     * @param  when When along the automation timeline to convert the ticks.
     * @return The duration in ticks.
     */
    timeToTicks(duration: Time, when: Time): Ticks;
    /**
     * Convert from the type when the unit value is BPM
     */
    protected _fromType(val: UnitMap[TypeName]): number;
    /**
     * Special case of type conversion where the units === "bpm"
     */
    protected _toType(val: number): UnitMap[TypeName];
    /**
     * A multiplier on the bpm value. Useful for setting a PPQ relative to the base frequency value.
     */
    get multiplier(): number;
    set multiplier(m: number);
}
export {};
