<?php

namespace App\Http\Controllers;

use App\Models\Configuration;
use App\Models\Pengumuman;
use App\Models\Queue;
use App\Models\QueueLog;
use Carbon\Carbon;
use Illuminate\Http\Request;

class SiswaController extends Controller
{
    public function index()
    {
        $today = now()->startOfDay();
        $queueLimit = (int) Configuration::where('key', 'queue_limit')->value('value');
        $jumlahAntrianHariIni = Queue::whereDate('tanggal_buka', $today->toDateString())->count();
        $remainingSlots = $queueLimit - $jumlahAntrianHariIni;
        return view("siswa.index", compact('queueLimit', 'jumlahAntrianHariIni', 'remainingSlots'));
    }

    public function petugas()
    {
        $today = now()->startOfDay();
        $queues = Queue::whereDate('tanggal_buka', $today->toDateString())->get();
        $queueLimit = (int) Configuration::where('key', 'queue_limit')->value('value');
        $jumlahAntrianHariIni = Queue::whereDate('tanggal_buka', $today->toDateString())->count();
        $remainingSlots = $queueLimit - $jumlahAntrianHariIni;
        
        return view("petugas.index", compact('queueLimit', 'jumlahAntrianHariIni', 'remainingSlots', 'queues'));
    }

    public function create(Request $request)
{
    $today = now()->startOfDay();
    $yesterday = $today->copy()->subDay();

    // Step 1: Delete queues from yesterday to keep the database clean
    Queue::whereDate('tanggal_buka', $yesterday)->delete();

    // Fetch the dynamic queue limit from the configuration
    $queueLimit = (int) Configuration::where('key', 'queue_limit')->value('value');

    // Step 2: Find and update queues from yesterday that exceeded the limit
    $yesterdayQueues = Queue::whereDate('tanggal_buka', $yesterday->toDateString())->get();

    foreach ($yesterdayQueues as $queue) {
        if ($queue->kuota > $queueLimit) {
            $queue->tanggal_buka = $today;
            $queue->save();
        }
    }

    // Step 3: Count today's queues after updating
    $jumlahAntrianHariIni = Queue::whereDate('tanggal_buka', $today->toDateString())->count();
    $remainingSlots = $queueLimit - $jumlahAntrianHariIni;

    if ($jumlahAntrianHariIni >= $queueLimit) {
        $nextDay = $today->copy()->addDay();

        // Find the next available day with less than the queue limit
        while (Queue::whereDate('tanggal_buka', $nextDay->toDateString())->count() >= $queueLimit) {
            $nextDay = $nextDay->addDay();
        }

        // Create a new queue for the next available day
        $newQueue = Queue::create([
            'tanggal_buka' => $nextDay,
            'tanggal_tutup' => null,
            'kuota' => Queue::whereDate('tanggal_buka', $nextDay->toDateString())->count() + 1
        ]);

        $nomorAntrian = $nextDay->format('ymd') . '-' . str_pad($newQueue->kuota, 3, '0', STR_PAD_LEFT);
    } else {
        // Create a new queue for today
        $newQueue = Queue::create([
            'tanggal_buka' => $today,
            'tanggal_tutup' => null,
            'kuota' => $jumlahAntrianHariIni + 1
        ]);

        $nomorAntrian = $today->format('ymd') . '-' . str_pad($newQueue->kuota, 3, '0', STR_PAD_LEFT);
    }

    // Log the queue creation
    $queueLog = new QueueLog();
    $queueLog->queue_id = $newQueue->id;
    $queueLog->status = 'Menunggu';
    $queueLog->tanggal_masuk = now();
    $queueLog->save();

    // Generate the queue card
    $kartuAntrian = view('kartu_antrian', [
        'queueLog' => $queueLog,
        'nomorAntrian' => substr($nomorAntrian, 7) // Hanya tampilkan bagian nomor, bukan tanggal
    ])->render();

    // Return the response
    return response()->json([
        'success' => true,
        'kartuAntrian' => $kartuAntrian,
        'queueLimit' => $queueLimit,
        'remainingSlots' => $remainingSlots
    ]);
}

public function list()
{
    $today = now()->startOfDay();
    $queues = Queue::whereDate('tanggal_buka', $today->toDateString())->get();

    return view('petugas.list', compact('queues'));
}


public function getQueueData()
{
    $today = now()->startOfDay();
    $queueLimit = (int) Configuration::where('key', 'queue_limit')->value('value');
    $jumlahAntrianHariIni = Queue::whereDate('tanggal_buka', $today->toDateString())->count();
    $remainingSlots = $queueLimit - $jumlahAntrianHariIni;
    $pengumuman = Pengumuman::first();

    return response()->json([
        'queueLimit' => $queueLimit,
        'remainingSlots' => $remainingSlots,
        'pengumuman' => $pengumuman,
    ]);
}


    public function printKasir(Request $request)
    {
        $kartuAntrian = $request->input('kartuAntrian');

        $printHtml = '
        <!DOCTYPE html>
        <html>
        <head>
            <title>Cetak Kartu Antrian</title>
            <style>
                body {
                    font-family: Arial, sans-serif;
                    margin: 0;
                    padding: 0;
                }
                .struk {
                    width: 200px;
                    text-align: center; /* Mengatur teks menjadi rata tengah */
                    font-family: "Courier New", Courier, monospace;
                    margin-left: 0; /* Mengatur margin-left menjadi 0 */
                }
                .nomor-antrian {
                    font-weight: bold;
                    font-size: 30px;
                }
            </style>
        </head>
        <body>
            <div class="struk">
                <center><big><span class="nomor-antrian">' . $kartuAntrian . '</span></big></center>
            </div>
        </body>
        </html>
        ';

        return response()->json(['kartuAntrian' => $printHtml]);
    }
    
  public function printDataKasir(Request $request)
{
    $queueId = $request->input('queue_id');
    $queue = Queue::find($queueId);
    $kuota = $queue->kuota;
    $nomorAntrian = $request->input('kuota');
    $tanggal_buka = $request->input('tanggal_buka');
    $tanggal_masuk = \Carbon\Carbon::parse($tanggal_buka);
    $hari = ['Minggu', 'Senin', 'Selasa', 'Rabu', 'Kamis', 'Jumat', 'Sabtu'];
    $bulan = ['Januari', 'Februari', 'Maret', 'April', 'Mei', 'Juni', 'Juli', 'Agustus', 'September', 'Oktober', 'November', 'Desember'];
    $hari_ini = $hari[$tanggal_masuk->dayOfWeek];
    $bulan_ini = $bulan[$tanggal_masuk->month - 1];
    $formattedDate = "{$hari_ini}, {$tanggal_masuk->day} {$bulan_ini} {$tanggal_masuk->year}";
    $printHtml = '
    <!DOCTYPE html>
<html>
<head>
    <title>Cetak Kartu Antrian</title>
    <style>
        body {
            font-family: Arial, sans-serif;
            margin: 0;
            padding: 0;
        }
        .card {
            position: absolute;
            top: 10px;
            left: 10px;
            background-color: transparent;
            font-family: "Courier New", Courier, monospace;
        }
        .card-title {
            font-size: 18px;
            font-weight: bold;
            margin-bottom: 5px;
        }
        .nomor-antrian {
            font-size: 48px;
            font-weight: bold;
            margin-bottom: 10px;
        }
        .tanggal {
            font-size: 18px;
            font-weight: bold;
            margin-bottom: 10px;
        }
        .important {
            font-size: 16px;
            color: red;
            font-weight: bold;
        }
    </style>
</head>
<body>
    <div class="card">
        <div class="card-body">
            <h4 class="card-title">Kartu Antrian PPDB</h4>
            <h5 class="card-title">SMKN 2 Kudus</h5>
            <div class="nomor-antrian">' . $kuota . '</div>
            <h5 class="tanggal">' . $formattedDate . '</h5>
            <h5 class="important">Jangan sampai kartu hilang!</h5>
        </div>
    </div>
</body>
</html>';
    return response()->json(['kartuAntrian' => $printHtml]);
}
}
