<?php

namespace App\Http\Controllers;

use App\Models\Pengumuman;
use App\Models\Queue;
use App\Models\QueueLog;
use App\Models\User;
use Illuminate\Http\Request;

class AdminController extends Controller
{
    public function index()
    {
        $totalAntrian = Queue::count();

        $antrianMenunggu = QueueLog::where('status', 'Menunggu')->count();
        $antrianDiproses = QueueLog::where('status', 'Diproses')->count();
        $antrianSelesai = QueueLog::where('status', 'Selesai')->count();

        $daftarAntrian = QueueLog::with('queue')
            ->select('queue_logs.*')
            ->take(5)
            ->get();

        return view('admin.index', compact('totalAntrian', 'antrianSelesai', 'antrianDiproses', 'antrianMenunggu', 'daftarAntrian'));
    }


    public function create()
    {
        return view('admin.user.create');
    }

    public function store(Request $request)
    {
        User::create([
            'name' => $request->name,
            'username' => $request->username,
            'level' => $request->level,
            'password' => $request->password,
        ]);

        return redirect()->route('Admin.users')->with('success', 'Penduduk berhasil ditambahkan');
    }

    public function getLatestQueueData()
{
    $today = now()->toDateString();
    $pengumuman = Pengumuman::first();

    $latestQueue = Queue::whereDate('tanggal_buka', $today)
        ->orderBy('kuota', 'desc')
        ->first();

    $antrianSekarang = QueueLog::join('queues', 'queue_logs.queue_id', '=', 'queues.id')
        ->join('users', 'queue_logs.user_id', '=', 'users.id')
        ->whereDate('queues.tanggal_buka', $today)
        ->where('queue_logs.status', 'Diproses')
        ->orderBy('queues.kuota', 'desc')
        ->select('queue_logs.*', 'users.name as operator_name', 'users.name as operator_full_name') // Tambahkan kolom 'name' dari tabel 'users'
        ->with('queue') // Eager load the queue relationship
        ->first();

    // Tambahkan kolom 'name' (nama lengkap operator) dari tabel 'users'
    $antrianSekarang['operator_full_name'] = $antrianSekarang->operator_full_name ?? null;

    $antrianSelanjutnya = QueueLog::join('queues', 'queue_logs.queue_id', '=', 'queues.id')
        ->whereDate('queues.tanggal_buka', $today)
        ->where('queue_logs.status', 'Menunggu')
        ->where('queues.kuota', '>', optional($antrianSekarang)->queue->kuota ?? 0)
        ->orderBy('queues.kuota', 'asc')
        ->select('queue_logs.*')
        ->with('queue') // Eager load the queue relationship
        ->first();

    $antrianSelesai = QueueLog::join('queues', 'queue_logs.queue_id', '=', 'queues.id')
        ->whereDate('queues.tanggal_buka', $today)
        ->where('queue_logs.status', 'Selesai')
        ->orderBy('queue_logs.id', 'desc')
        ->with('queue') // Eager load the queue relationship
        ->first();

    $operators = [];
    for ($i = 1; $i <= 13; $i++) {
        $operator = QueueLog::join('queues', 'queue_logs.queue_id', '=', 'queues.id')
            ->join('users', 'queue_logs.user_id', '=', 'users.id')
            ->whereDate('queues.tanggal_buka', $today)
            ->where('queue_logs.user_id', function ($query) use ($i) {
                $query->select('id')
                    ->from('users')
                    ->where('name', 'Operator ' . $i)
                    ->limit(1);
            })
            ->orderBy('queue_logs.created_at', 'desc')
            ->select('queue_logs.*', 'users.name as operator_name', 'users.name as operator_full_name') // Tambahkan kolom 'name' dari tabel 'users'
            ->with('queue') // Eager load the queue relationship
            ->first();
        $operators['operator' . $i] = $operator;
    }

    return response()->json(array_merge([
        'latestQueue' => $latestQueue,
        'antrianSekarang' => $antrianSekarang,
        'antrianSelanjutnya' => $antrianSelanjutnya,
        'antrianSelesai' => $antrianSelesai,
    ], $operators));
}



    public function user(Request $request)
    {
        $search = $request->input('search');
        $user = User::when($search, function ($query) use ($search) {
            $query->where('username', 'like', '%' . $search . '%');
        })
            ->paginate(10);

        return view('admin.user.index', compact('user'));
    }

    public function userAdminEdit($user_id)
    {
        $user = User::findOrFail($user_id);

        $data = [
            'name' => $user->name,
            'username' => $user->username,
            'level' => $user->level,
            'password' => $user->password,
        ];

        return response()->json($data);
    }

    public function adminUpdate(Request $request, $user_id)
    {
        $user = User::findOrFail($user_id);

        $user->update([
            'name' => $request->name,
            'username' => $request->username,
            'level' => $request->level,
            'password' => $request->password,
        ]);

        return redirect()->route('Admin.users')->with('success', 'User berhasil diperbarui');
    }

    public function monitor()
    {
        $pengumuman = Pengumuman::first();
        $antrianSekarang = QueueLog::where('status', 'Diproses')->orderBy('id')->first();

        $antrianSelanjutnya = QueueLog::where('status', 'Menunggu')
            ->where('id', '>', optional($antrianSekarang)->id ?? 0)
            ->orderBy('id')
            ->first();

        $antrianDiproses = QueueLog::where('status', 'Diproses')->count();
        $antrianSelesai = QueueLog::where('status', 'Selesai')->count();
        $totalAntrian = Queue::count();

        return view('admin.monitor', compact('totalAntrian', 'antrianSekarang', 'antrianSelanjutnya', 'antrianSelesai', 'pengumuman'));
    }

    public function edit($id)
    {
        $antrian = QueueLog::findOrFail($id);
        $nama_antrian = $antrian->queue->nama_antrian;

        return response()->json([
            'id' => $antrian->id,
            'nama_antrian' => $nama_antrian,
            'status' => $antrian->status,
        ]);
    }

    public function update(Request $request, $id)
    {
        $request->validate([
            'status' => 'required|in:Menunggu,Diproses,Selesai',
            'nama_antrian' => 'required|string|max:255',
        ]);

        $antrian = QueueLog::findOrFail($id);
        $antrian->status = $request->status;
        $antrian->save();

        // Perbarui nama_antrian jika ada dalam request
        if ($request->has('nama_antrian')) {
            $antrian->queue->nama_antrian = $request->nama_antrian;
            $antrian->queue->save();
        }

        return redirect()->route('Admin.dashboard')->with('success', 'Data antrian berhasil diperbarui.');
    }

    public function updateStatus($kuota)
    {
        // Mendapatkan instance dari model QueueLog berdasarkan kuota
        $queueLog = QueueLog::whereHas('queue', function ($query) use ($kuota) {
            $query->where('kuota', $kuota);
        })->first();

        if (!$queueLog) {
            return response()->json([
                'message' => 'No query results for model QueueLog with kuota ' . $kuota,
            ], 404);
        }

        // Memperbarui status antrian
        $queueLog->status = 'Diproses';
        $queueLog->save();

        return response()->json([
            'message' => 'Status updated successfully for QueueLog with kuota ' . $kuota,
            'queueLog' => $queueLog,
        ]);
    }

    public function destroy($user_id)
    {
        $user = User::findOrFail($user_id);
        $user->delete();

        return redirect()->route('Admin.users')->with('success', 'Penduduk berhasil dihapus');
    }

    public function delete($kuota)
    {
        // Mendapatkan instance dari model QueueLog berdasarkan kuota
        $queueLog = QueueLog::whereHas('queue', function ($query) use ($kuota) {
            $query->where('kuota', $kuota);
        })->first();

        if (!$queueLog) {
            return response()->json([
                'message' => 'No query results for model QueueLog with kuota ' . $kuota,
            ], 404);
        }

        Queue::count() - 1;

        $queueLog->delete();

        return redirect()->route('Admin.dashboard')->with('success', 'Data antrian berhasil dihapus.');
    }

    public function deleteAll()
    {
        QueueLog::truncate();
        Queue::query()->delete();

        return response()->json('Data antrian berhasil dihapus');
    }


    public function riwayatAll()
    {
        $daftarAntrian = QueueLog::with('queue')
            ->select('queue_logs.*')
            ->paginate(10);;

        return view('admin.riwayat.allRiwayat', compact('daftarAntrian'));
    }

    public function riwayatTunggu()
    {
        $daftarAntrian = QueueLog::with('queue')
            ->where('status', 'Menunggu')
            ->paginate(10);

        return view('admin.riwayat.riwayatTunggu', compact('daftarAntrian'));
    }

    public function riwayatProses()
    {
        $daftarAntrian = QueueLog::with('queue')
            ->where('status', 'Diproses')
            ->paginate(10);

        return view('admin.riwayat.riwayatProses', compact('daftarAntrian'));
    }

    public function riwayatSelesai()
    {
        $daftarAntrian = QueueLog::with('queue')
            ->where('status', 'Selesai')
            ->paginate(10);

        return view('admin.riwayat.riwayatSelesai', compact('daftarAntrian'));
    }
}
